/*
 * JavaTree - 3D tree generation library and demo GUI
    Copyright (C) 2012 Luke Wallin

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package javatree;

import LukesBits.*;
import java.util.ArrayList;

/*
 * Representing angles better:
 * 
 * want everything to be truly relative, so altering one angle means the others don't need to be changed,
 * but the rest of the branch will change with that first angle.
 * Euler angles are probably the most elegant solution, but fiddly to comprehend and possibly implement.
 *
 * I'm thinking of using rotating matrices atm.
 * Each branch has a rotation matrix, the trunk has a unit vector.
 * 
 * To get a rotation matrix to rotate from one vector to another, I *think* a cross product will work
 * the cross product will give an axis of rotation and the angle.
 * 
 * Either store an axis of rotation and an angle, or store the rotation matrix, or find nice way of converting
 * either of these into euler angles - I suspect this might be fiddly even though it'd be the most
 * elegant way of doing it.
 * 
 * 
 * might also want to consider making the axis right handed at some point?
 * 
 * 
 * 
 */




//own class seems bit OTT atm, but might have future benefits.

/**
 *
 * @author Luke, 
 * 
 * do we want a branch interface so we can have different types of branches in the same tree?
 */
public class Branch implements IBranch {
    //private Vector angle;//relative to parent  NOTE this hsould be perpendicular to the parent angle (angle.parentAngle = 0)
    
    private Vector stump,tip,absoluteAngle;
    
    private double length;
    private double radius;
    //how far along the parent the start of our position is
    //private double parentFraction;
    private Tree tree;//might be useful to have the properties of the tree avaliable?
    private IBranch parent;
    private ArrayList<Branch> children;
    
    private double growthRate, growthRateChange;
    //rotate the parent by this to get our angle
    private Matrix wantAngle;
    private Matrix angle;
    
    //private ArrayList<FutureChildBranch> futureChildren;
    
    //have we craeted the branch which continues this branch? (the one at the very end with a similar angle)
    private boolean continuedBranch;
    
    //how long to crow before kinking, maxlength is how long before we branch into a new segmenet, which can have its own hcildren
    private double kinkLength,maxLength;
    
    private boolean trunk;
    
    
    public Branch (Tree _tree,IBranch _parent, Matrix _rotated, boolean _trunk){//Vector _angle, ){//, double _radius){
        tree=_tree;
        parent=_parent;
        //angle=_angle;
        //parentFraction=_parentFraction;
        length=0;
        //our rotation matrix!
        wantAngle=_rotated;
        angle=wantAngle.copy();
        children = new ArrayList<Branch>();
        
        growthRate=tree.getGrowthRate();
        growthRateChange = tree.getGrowthRateChange();
        
        //this branch is the continuation of its parent
        trunk=_trunk;
        
        //kinkLength = tree.getLengthOfBranchSegment();
        maxLength = tree.getLengthOfBranch();
        
        calculateVectors();
        
        //futureChildren = new ArrayList<FutureChildBranch>();
        //TODO control this from the tree
        //tipRadius = _radius;
    }
    
    public double getParentRadius(){
        return parent.getRadius();
    }
    
    public double getLength(){
        return length*tree.getScale();
    }
    
    public double getRadius(){
        if(children.size() > 0){
            double childrenArea=0;
            for(Branch c : children){
                
                double childRadius = c.getRadius();
                
                //area of a circle = pi*r^2
                childrenArea+=childRadius*childRadius*Math.PI;
            }
            
            //the area of this branch is the same as the area of all the child branches, so the radius is
            //r= sqrt(area/pi)
            
            double radius = Math.max(tree.getTipRadius(),Math.sqrt(childrenArea/Math.PI));
            return radius;//*tree.getScale();
        }else{
            //TODO link this up so the radius grows to tipRadius gradually until a child is spawned
            return tree.getTipRadius() * Math.min(1.0, length/maxLength);//*tree.getScale();
        }
    }
    
    public double getMass(){
        
        double childrenMass=0;
        
        for(Branch b : children){
            childrenMass+=b.getMass();
        }
        
        double ourMass = Math.PI*radius*radius * length * tree.getDensity() / Math.sqrt(tree.getScale());//remove effects of scaling the tipradius for mass
        
        //not sure if this needs to be scaled or not :/
        return (childrenMass + ourMass);//*Math.pow(tree.getScale(),3);
    }
    
    
    
    /*
    public ArrayList getChildren(){
        return children;
    }
    
    public Vector getAngle(){
        
        return angle;
    }*/
    
    private void calculateVectors(){
        //TODO, not reutrn unit vector to speed things up?
        
        //stump = parent.getFraction(parentFraction);
        stump = parent.getTip();
        //absoluteAngle = parent.getAbsoluteAngle().add(angle).getUnit();
        
        absoluteAngle = parent.getAbsoluteAngle().rotate(angle);
        
        tip = stump.add(absoluteAngle.multiply(length*tree.getScale()));
        
        radius = getRadius();
       
    }
    
    @Override
    public Vector getAbsoluteAngle(){
        return absoluteAngle;
    }
    
    public Vector getAbsoluteWantAngle(){
        return parent.getAbsoluteAngle().rotate(wantAngle);
    }
    
    public Vector getTip(){
        return tip;
    }
    
    //this appears to slow it down a lot
    //get a certain fraction along this branch
//    public Vector getFraction(double fraction){
//         return stump.add(absoluteAngle.multiply(length*fraction));
//    }
    
    public Vector getStump(){
        //return parent.getTip();
        return stump;
    }

    private void droop(double time){
        //new idea:  force from wantposition + force from gravity + light?
        //the new idea works :D
        
        //improvement to this idea:
        //rotate wantAngle very very slightly in direction of the droopangle - will need to be time dependant!
        
        
        
         //for drooping stuff:
        //http://en.wikipedia.org/wiki/Deflection_(engineering)
        //gives a static equation for the angle of the free end of a cantilever 
        //angle of deflection = ( F * L^2 ) / ( 2 * E * I )
        //F= force on tip of beam,
        //L = length of beam
        //E = modulus of elasticity (higher = stiffer)
        //I = area moment of intertia (circular cross section = (pi*r^4)/4
        //it's going to be a bit of a kludge using this for a branch which is attached to another branch, and also not horizontal to start with,
        //but it might help
        //almost worth using a real physics engine...
        
        //idea: use force dotted with normal to the branch?  so we're essentially rotating the cantilever problem 
        //and only using the force that is perpendicular to the beam
        
        //want forces normal to the branch so we can treat it like a cantilever
        
        Vector absWantAngle = getAbsoluteWantAngle();
        
        //light = l, gravity = g
        Vector l = tree.getLight();
        Vector g = tree.getGravity();
        double E = tree.getElasticity();
        //double I = Math.pow(radius, 4)*Math.PI/4;
        double I = Math.pow(radius, 4)*100*tree.getElasticity();
        
        double m = getMass()*0.01;
        
        Vector f = l.copy();
        f = f.add(g.multiply(m));
        f = f.add(absWantAngle.multiply(I));//

        //want the branch to point at f
        f= f.getUnit();
        
        //might need to be other way around?
        Vector cross = parent.getAbsoluteAngle().cross(f);//f.cross(parent.getAbsoluteAngle());

        //Matrix newRotation = Vector.rotationMatrix(newAbsoluteAngle, Math.asin(newAbsoluteAngle.getMagnitude()));
        double crossAngle = Math.asin(cross.getMagnitude());
        Matrix newRotation = Vector.rotationMatrix(cross, crossAngle);
        
        //TODO check that the *time does actually do waht I want : that a large timestep produces the same results as lots of small timesteps.
        Matrix smallRotation = Vector.rotationMatrix(cross, crossAngle*0.003*time);
        
        //rotate the wantAngle a small amount in the direction as the droopangle
        wantAngle = wantAngle.times(smallRotation);
        
        //this slowly bends the angle there:
        angle = newRotation;
        //do we need some iteration here?
    }
    
    //if trunk is true, this child branch is the continuation of the current branch
    private void growBranch(double maxAngle, boolean _trunk){
        //distance OUT - towards the 'cone'
            //angle round - how it rotates
            
            
            /*
             * 
             * 
             * | \
             * |  \
             * |   \
             * |    \
             * |     \
             * |  /\  \ - cone
             *     |
             *      theta goes in there
             * other angle rotates aroudn the branch
             * 
             */
            
            
            /*
             * new, better, faster, simpler idea!!
             * 
             * get a vector which is normal to this vector
             * 
             * dot product + want unit vector means I only have to choose a value for one variable (carefully to make sure not parallel)
             * and then with a psuedorandom function I can get a vector which is normal and points in a random direction!!
             * 
             * add a small quanitity of this vector to the first vector and tada!
             * 
             * hurrah :D
             * 
             * 
             */
            
            //get a random vector which is normal to this direction
            Vector normal = absoluteAngle.randomNormal(tree.getRandom());
            
            Vector newAngle = normal.multiply(maxAngle);
            //newAngle now (can) points slightly away form this branch 
            Vector newAbsoluteAngle = absoluteAngle.add(newAngle).getUnit();
            
            //might need to be other way around?
            Vector cross = newAbsoluteAngle.cross(absoluteAngle);
            //Matrix newRotation = Vector.rotationMatrix(newAbsoluteAngle, Math.asin(newAbsoluteAngle.getMagnitude()));
            Matrix newRotation = Vector.rotationMatrix(cross, Math.asin(cross.getMagnitude()));
            
            Branch b = new Branch(tree,this,newRotation,_trunk);
            
            children.add(b);
            
            //TODO, does tree need all the branches like this?
            tree.addBranch(b);
            
            //b.generateRandomChildren();
    }
    
    public void grow(double time){
        length+=growthRate*time;
        
        growthRate*=1 + growthRateChange*time;
        
        //gravity droop - calculates the actual angles all the branches are at as a 
        //result of gravity and light.
        //also slightly alters the wanted angle.
        droop(time);

        
        if(tree.growRareBranch()){
            growBranch(tree.getBranchVariation(),false);
        }
        
        if(length>maxLength && !continuedBranch){
            
            continuedBranch=true;
            
            //new branch at end ofhtis branch, similar dir
            //if(tree.getRandomDouble() > 0.1){
            
                //always continue the branch
                growBranch(tree.getTrunkVariation(),true);
            //}
            
                if(tree.growBranch()){
                    growBranch(0.8,false);
                }
                
        }
        
        //now re-calcuate all the vector stuff
        
        calculateVectors();
    }
    
    //use the properties of the tree to queue up children which should be grown later
//    public void generateRandomChildren(){
//        futureChildren = new ArrayList<FutureChildBranch>();
//        int numChildren = tree.getNumChildren();
//        for(int i=0;i<numChildren;i++){
//            double howFar = tree.getWhereChild();
//            futureChildren.add(new FutureChildBranch(howFar));
//        }
//    }
    
    //grow these children later
//    public void growTheseChildren(ArrayList<FutureChildBranch> _futureChildren){
//        futureChildren=_futureChildren;
//    }

    @Override
    public boolean hasChildren() {
        return !children.isEmpty();
    }

    @Override
    public Vector getParentAbsoluteAngle() {
        return parent.getAbsoluteAngle();
    }

    @Override
    public boolean isTrunk() {
        return trunk;
    }

    @Override
    public IBranch getParent() {
        return parent;
    }
    
}
class FutureChildBranch{
    //make this child start to grow after the branch is this long
    public double afterLength;
    
    public FutureChildBranch(double _afterLength){
        afterLength = _afterLength;
    }
    
}