/*
 * JavaTree - 3D tree generation library and demo GUI
    Copyright (C) 2012 Luke Wallin

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package javatree;

import java.awt.*;
import java.util.*;
import java.awt.geom.Line2D;
import javax.imageio.*;
import java.io.*;
import java.awt.image.*;
import java.awt.*;
import LukesBits.Vector;

/**
 *
 * @author Luke
 * 
 * control system idea:
 * seperate window, or control panel in same window for all the options.
 * sliders where possible
 * animatino for changing from one end ofht escale to another?
 * 
 * still worth doing a glicko comparison of pairs of trees to home in on properties to generate certain types of tree.
 * 
 * 
 * structure: all properties that apply to all branches should be set with a public method from the tree, no point passing around loads of values to each branch
 * 
 * ideas being trialled:
 * 
 * all branches grow at the same rate - where rate is an absolute rather than relative to the size of the branch.
 * - might re-visit this idea?
 * 
 * probability of when to grow a new branch
 * 
 * 
 * idea: angle is the angle a branch wants to be at, have strength or similar, combined with weight and gravity for where it actually is
 * that way can do stuff like willow and birch, where the ends dangle
 * properties: wantAngle + strength, combined with how much weight is on the branch?  strength improves with radius?
 * 
 * idea: use cantalever maths for working out the final angle?
 * 
 * 
 * idea: adjustment for end thickenss of branches, so can still have spindly trees and thick trees like oak
 * 
 * 
 * idea: two types of branch class - 'end' branches which can only grow leavces ,adn full branches which can grow more branches 
 * end branches can die (light dependent?)
 * 
 * 
 * http://en.wikipedia.org/wiki/Apical_dominance
 * idea for this: have a trunkstrength argument - this can reduce the max angle for the continuing branch, and maybe reduce the the growth rate of
 * non trunk branchlets?
 * 
 * 
 * 
 * idea that might work for a willow: only grow branches when strong enough?  or above a certain radius?
 */
public class Tree{// implements Runnable {

    /**
     * @param args the command line arguments
     */
    
    //private double stiffness;//how much branches droop due to gravity/affected by wind, etc
    private double growthRate,growthRateChange;
    private Vector gravity;
    private Vector light;
    private Vector wind;
    
    private Vector pos,angle;
    
    //how much to scale the size of the tree by
    private double scale;
    
    private ArrayList<IBranch> branches;
    private Random random;
    private double time;
    private long seed;
    //private TreeWindow window;
    private Trunk trunk;
    private Camera camera;
    private double tipRadius,density,elasticity,branchChance;
    
    private double cameraAngle;
    
    private double muBranchLength,sigmaBranchLength,minBranchLength;
    
    private double muNumChildren,sigmaNumChildren;

    private boolean drawAxis,colourful;
    
    public double getScale(){
        return scale;
    }
    
    public void setScale(double _scale){
        scale=_scale;
        //to force a recalculation of all the vectors
        //bit hacky...
        grow(0.0);
    }
    
    public Vector getPos(){
        return pos;
    }
    
    public Vector getAngle(){
        return angle;
    }
    
    public double getGrowthRate() {
        return growthRate;
    }
    
    public ArrayList<IBranch> getBranches(){
        return branches;
    }
    
    public double getGrowthRateChange(){
        return growthRateChange;
    }

//    public double getStuffness() {
//        return stiffness;
//    }

    public Vector getGravity(){
        return gravity;
    }
    
    public Vector getLight(){
        return light;
    }
    
    public double getElasticity(){
        //wolfram says 9.4Gpa for average wood, 11Gpa for oak
        //realistic numbers are no longer being used.
        return elasticity;
    }
    //how many children should each branch grow?  atm doesn't include continuing itself as a branch
    public int getNumChildren(){        
        return (int)Math.round(random.nextGaussian()*sigmaNumChildren + muNumChildren);
    }
    
    public double getWhereChild(){
        return random.nextDouble()*100;
    }
    
    public double getTipRadius(){
        return tipRadius*scale;
    }
    
    public double getBaseRadius(){
        return trunk.getRadius();
    }
    
    //branches will have getNumChildren per this length
    public double getLengthOfBranch(){
        return Math.max(random.nextGaussian()*sigmaBranchLength + muBranchLength, minBranchLength);
    }
    
    //does a branch grow another branch?
    public boolean growBranch(){
        return random.nextDouble() < branchChance;
    }
    
    //does a branch grow completely randomly?
    public boolean growRareBranch(){
        return random.nextDouble() < branchChance*0.001;
    }
    
    //how much the trunk can vary in angle, 1=upto 90deg, 0=perfectly straight.
    public double getTrunkVariation(){
        return 0.1;
    }
    //same but for branches
    public double getBranchVariation(){
        return 0.8;
    }
    
    public double getDensity(){
        return density;
    }
    
    //find the highest point of the tree in the line of angle of the tree
    public double getMaxHeight(){
        double maxHeight=0;
        
        for(IBranch b : branches){
            //find how far in the direction of the angle of the tree the end of this branch is
            double distance = b.getTip().subtract(pos).dot(angle);
            if(distance>maxHeight){
                maxHeight=distance;
            }
        }
        
        return maxHeight;
    }
    
    //find how far the furthest branch is from teh base of tree
    public double getMaxSize(){
        double maxHeight=0;
        
        for(IBranch b : branches){
            //find how far in the direction of the angle of the tree the end of this branch is
            double distance = b.getTip().subtract(pos).getMagnitudeSqrd();//.dot(angle);
            if(distance>maxHeight){
                maxHeight=distance;
            }
        }
        
        return Math.sqrt(maxHeight);
    }
    
    public Tree() {
        this(new TreeParameters());
    }
    
    public Tree(TreeParameters params){
        gravity = params.gravity;
        light = params.light;
        wind = params.wind;
        
        pos=params.pos;
        angle=params.angle;
        scale=params.scale;
        
        seed = params.seed;
        
        density = params.density;
        elasticity = params.elasticity;
        tipRadius = params.tipRadius;
        
        branchChance = params.branchChance;
        muBranchLength = params.muBranchLength;
        sigmaBranchLength = params.sigmaBranchLength;
        minBranchLength = params.minBranchLength;
        
        muNumChildren = params.muNumChildren;
        sigmaNumChildren = params.sigmaNumChildren;
        
        growthRate = params.growthRate;
        
        drawAxis = params.drawAxis;
        colourful = params.colourful;
        
        growthRateChange = params.growthRateChange;
        
        initTree();
    }
    
   

    private void initTree(){
        random=new Random(seed);
        //random = new Random(324);

        branches = new ArrayList<IBranch>();

        trunk = new Trunk(this, pos, angle);
        
        addBranch(trunk);

        time = 0;
        cameraAngle=0.1;
        
        //camera = new Camera(new Vector(-5500,-5500,0), new Vector(1,1,0.21), 20, 10);
        camera = new Camera(new Vector(-5500,-5500,0), new Vector(1,1,0.21), 20, 10);
        camera.setupRez(640, 480);
    }
    
    public synchronized void saveImage(String imageName){
        saveImage(imageName,1000,1000);
    }
    
    //bit of a dirty hack making this synchronized?
    public synchronized void saveImage(String imageName, int width, int height){
        try {
                BufferedImage b = new BufferedImage(width,height,BufferedImage.TYPE_INT_RGB); /* change sizes of course */
                Graphics2D g = b.createGraphics();
                Draw(g, width,height);
                
                
                ImageIO.write(b, "png", new File(imageName));
            } catch (Exception e) {
            }
    }

    public double getRandomDouble() {
        return random.nextDouble();
    }

    public Random getRandom() {
        return random;
    }
    
    public void setBranchLengths(double mu,double sigma, double min){
        muBranchLength = mu;
        sigmaBranchLength = sigma;
        minBranchLength = min;
    }
    
    public synchronized void grow(double _time) {
        //grow for time.
        //this isn't used yet:
        time+=_time;
        //done like this as the branches array can grow while this happens
        int currentBranches = branches.size();

        for (int i = 0; i < currentBranches; i++) {
            branches.get(i).grow(_time);
        }

    }

    public void addBranch(IBranch _branch) {
        branches.add(_branch);
    }

    /*
     * idea for simple 3D camera wherever I want:
     * 
     * (nick the code from the raytracer):
     * 
     * there is a camera, which is a position vector, and a screen, which has 4 corners.
     * 
     * 
     * 
     */
    
    public synchronized void rotateCamera(double a){
        cameraAngle+=a;
    }
            
            
    public synchronized void Draw(Graphics g1, int width, int height) {
        
        //boolean colourful = true;
        
        double angle = cameraAngle;
        double distance = 10000;
        
        //works quite well for a high up view down:
//        Vector camLoc = new Vector(distance*Math.cos(angle), distance*Math.sin(angle), distance/2);
//        Vector camPoint = camLoc.multiply(-1.0).getUnit().add(new Vector(0,0,0.23));
        
        Vector camLoc = new Vector(distance*Math.cos(angle), distance*Math.sin(angle), distance/4);
        Vector camPoint = camLoc.multiply(-1.0).getUnit().add(new Vector(0,0,0.2));
        
        camera.setPosAndDir(camLoc, camPoint);
        camera.setupRez(width, height);

        Graphics2D g = (Graphics2D) g1;
        //Graphics g = g1;

        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

        g.setColor(Color.black);
        if(colourful){
            //pale sky blue
            g.setColor(new Color(153,221,255));
        }
        g.fillRect(0, 0, width, height);

        //get a point far in the distance
        
        
        if(colourful){
            Vector farAway=camera.ThreeDTo2D(new Vector(100000*Math.cos(angle+Math.PI), 100000*Math.sin(angle+Math.PI), 0));
            //dark green
            g.setColor(new Color(50,150,10));
            g.fillRect(0, farAway.getRoundedY(), width, height-farAway.getRoundedY());
        }
        g.setColor(Color.white); 
        //g.drawString("angle:"+cameraAngle+" camera: "+camera.getPos(),30, 50);
        //double mass = trunk.getMass();
        //g.drawString("tree mass = "+(Math.round(mass*100.0)/100.0),30, 50);
        g.drawString("tree radius = "+(Math.round(trunk.getRadius()*100.0)/100.0),30, 80);
        g.drawString("Seed: "+seed,30, 50);
        
        //int offsetX = Math.round(width / 2);
        //int offsetY = Math.round(height);

        //g.translate(offsetX, offsetY);
        //g.translate(50, 50);
        //g.scale(0.01, 0.01);

        //ignoring y, so drawing x along x axis and the '3d z axis' along the '2d y axis'
        
        if(drawAxis){
            //x axis
            g.setColor(Color.red);
            draw3DLine(g,new Vector(0,0,0), new Vector(100,0,0));

            //y axis
            g.setColor(Color.yellow);
            draw3DLine(g,new Vector(0,0,0), new Vector(0,100,0));

            //z axis
            g.setColor(Color.green);
            draw3DLine(g,new Vector(0,0,0), new Vector(0,0,100));
        }
        
        
        if(colourful){
            g.setColor(new Color(150, 75, 0));
        }else{
            g.setColor(Color.white);
        }
        for (IBranch b : branches) {
            
            //Vector start = camera.ThreeDTo2D(b.getStump());
            //Vector end = camera.ThreeDTo2D(b.getTip());

            /*nt x1=start.getRoundedX();
            int y1=-start.getRoundedZ();
            int x2=end.getRoundedX();
            int y2= -end.getRoundedZ();*/
            //g.drawLine(x1,y1 , x2,y2);

            //g.draw(new Line2D.Double(start.x, -start.z, end.x, -end.z));
            g.setStroke(new BasicStroke((float)b.getRadius()*2,BasicStroke.CAP_ROUND,BasicStroke.JOIN_ROUND));
            //g.draw(new Line2D.Double(start.x, start.y, end.x, end.y));
            draw3DLine(g,b.getStump(),b.getTip());
            //g.
        }

        
        
    }
    public String toString(){
        return "Seed:"+seed+", gravity:"+gravity+", light:"+light+", tipRadius:"+tipRadius+", ";
    }
    private void draw3DLine(Graphics2D g,Vector start, Vector end){
        Vector start3D = camera.ThreeDTo2D(start);
        Vector end3D = camera.ThreeDTo2D(end);
        
        g.draw(new Line2D.Double(start3D.x, start3D.y, end3D.x, end3D.y));
        //g.drawLine(start3D.getRoundedX(), start3D.getRoundedY(), end3D.getRoundedX(), end3D.getRoundedY());
    }
}
